/**
 *
 * \file        dm_dmnet_slave.hpp
 *
 * \brief       header file for DM Net Slave
 * 
 * \author      Pranav Bhargava
 *
 * \date        03/26/2009
 *
 */
#ifndef _DM_DMNET_SLAVE_HPP_
#define _DM_DMNET_SLAVE_HPP_

#include "dm_cresnet.h"
#include "dm_errors.h"
#include <string.h>
#include "dm_os.h"
#include "dm_uart_cresnet_master.h"
#include "dm_console.h"
#include "memorymanager.h"
#include <stdlib.h>
#include <stdio.h>

#include "dm_CresnetHardware.hpp"
#include "dm_OsEvent.hpp"
#include "CresnetStruct.h"
#include "HwInterf.h"
#include "dm_NetMapper.hpp"
#include "Cresnet.h"
#include "Com3xDriver.h"
#include "CSerialInterrupts.h"
#include "dm_project_video.h"
#include "DmEncapChecksumHelper.h"

#define SLAVE_RX_LISR_SET_RX_WAIT_TIMEOUT()  
#define SLAVE_RX_LISR_CLR_RX_WAIT_TIMEOUT()  

#define DMNET_SLAVE_TX_PIPE_SIZE 	5000 // Queue size for the DMLink Send
#define DMNET_SLAVE_RX_PIPE_SIZE	5000 // Queue size for the DMLink Receive
#define	MAX_DMNET_CARD_VERSION_STRING_LENGTH	200 // array size for the DM Input card version

// #pragma pack()
class CDMNetSlavePortLISRHandler;
class CDMNetSlaveTimerLISR ;
class CDMNetSlaveRxHISR;
class CDMNetSlaveRxTask;
class Uart_16C684;


typedef struct {
	int m_iNumBreakInterrupts;
	int m_iNumRxRdyInterrupts;
	int	m_iNumRxToutInterrupts;
	int	m_iNumDataRdyWithError;
	int m_iNumDataBytesWithoutSync;
	int m_iNumHISRFired;
	int m_iNumTokensRcvd;
	int m_iNumPktsRcvd;
	int m_iNumTokensSent;
	int m_iNumNotRespondingToPoll;
	int m_iNumPktsSent;
	int m_iNumIgnore;
	int m_iNumDrops;
	int m_iNumCharAdded;
	int m_iNumHISRInvokedForTokenPass;
	int m_iNumInvalidState;
	int m_iNumNotOurId;
	int m_iNumNotShippedUp;
	int m_iNumId02;
	int m_iNumPNpOrBroadCast;	
}DMNET_STATS;


class CDMUartCresnetSlave: public Uart_16C684  
{
    private:
        void   CheckPointersAfterInit(void) ;
		CREST_ALL_PACKETS	m_sEncapsulatedPacket;
		CREST_ALL_PACKETS m_sHelperOnlinePacket;
		CREST_ALL_PACKETS m_sHelperOfflinePacket;
		void InformDeviceOfEndpointOnline();
		void InformDeviceOfEndpointOffline();
        BOOL m_bTxEnable;

    public:
   
        CColdfire_BaseTimer         *m_pTimer;   // network mode and timing parameters
        CDMNetSlavePortLISRHandler  *m_pSlaveLISRHandler;	// DM Net Slave port LISR Handler
        CDMNetSlaveTimerLISR   		*m_pTimerLISR;			// DM Net Slave Timer LISR
        CDMNetSlaveRxHISR      		*m_pRxHISR;				// DM Net Slave RX HISR
        CDMNetSlaveRxTask           *m_pDMNetLinkRxTask ;	// DM Net Slave Rx task for packets coming off the DMLINK
        CIntRingBuffer          	*m_RxIntBuf ;			// Stores Incoming bytes
        CLinearBuffer           	*m_pTxBuf ;  			// init in SlaveLISR
        // Pipe                    *m_pTxDMNetRcvPipe;		// Pipe to hold packets coming off the DMNET
        Pipe                    	*m_pTxDMNetSendPipe;	// Pipe to hold packets being sent to the DMNET
		CDMNetMapper				*m_pDMNetMapper;		// Pointer to the DMNet Mapper class for this UART
        
    protected:
                               // coldfire specific
        void WriteClockSelReg(unsigned char val)  { ; } ;
        void WriteAuxCntrlReg(unsigned char val)  { ; }  ;
        void WriteInterruptVector(unsigned char val)  { ; }  ;
        BYTE ReadInputPortChangeReg(void)  { return 0; }  ;
        BYTE ReadInterruptStatus(void)  { return 0; } ;

    public:                             // coldfire specific
        void GrabBus(void) { DisableRxInt() ; En485_Tx() ; }  ;
        void DropBus(void) { En485_Rx(); EnableRxInt() ; }   ;

    public:                      // generic UART functions

        CDMUartCresnetSlave( int ,ADDRESS, DmEncapChecksumHelper* pHelper = 0) ;
        virtual ~CDMUartCresnetSlave() ;

        // low level calls
        BYTE RxRead(void) { return (exar_port->RHR_RG); };
        BYTE StatRead(void) { return (exar_port->ISR_RG); };

        // high level calls
		void StartBreak(void) { exar_port->LCR_RG |= LCR_BRK ; } ;
        void StopBreak(void) { exar_port->LCR_RG &= ~LCR_BRK ; } ;
#if defined (MPS_DM_BUILD)
        void Reconfigure( BOOL bEnable );
#endif        
		DMNET_STATS				m_sStats;

    public:                             // cresnet and slave specific
        CRESNET_HOST_STATE_ENUM m_iNetMode ;   // what it's doing now: sending token, wait on rcv, rcv, etc...
        int     m_iCresnetID;
        int     m_bJustPowered ;
        int     m_bIdRequest;
        int     m_iPrevCommand;
        int     m_bGotSync;
        int     m_iStatus;              //
        int     m_iRespondPpnMask;              //
        int     m_iCnetIdValid ;
		UINT8 	m_StreamId;				// Stores the stream Id
		BOOL 	m_bEndPointOnline;		// Indicates whether the endpoint is ONLINE or not ....
		BOOL 	m_bRespondToPoll;		// Indicates whether we should respond to POLLS

		void  ProcessIncomingDMLinkCmd(CREST_ALL_PACKETS *pack) ;

    public:                             // PnP and cresnet members
		UINT8 m_OfflineTimeout;

        void DisableRx(void) 
        { 
            exar_port->MCR_RG |= MCR_RTS ;
        } ;
        void EnableRx(void) 
        { 
           exar_port->MCR_RG &= ~MCR_RTS ;
        } ;
      
		// DMNetMapper Interface functions
		UINT8 IsUartCrestnetSlaveQueueFull(UINT8 bStream);
		void MidpointOnlineTimer(void);
		CREST_ALL_PACKETS * MidpointGetNextPacket(void);
		void MidpointReleaseNextPacket(UINT8 *pBuffer);
		void MidpointSetOnLineStatus(UINT8 State);
		void SendPacketToTxForEndpoint(UINT8 *pPacket);
		bool IsMidpointOnline(void);
		UINT32 UartCresnetMidpointRxIsr(void);
		UINT32 UartCresnetSlaveRxIsr(void);
        void UartCresnetSlaveQueueTxPacket(UINT8 bStream, void *packet, UINT32 packet_size);
		void GetFirmwareVersion(char *pFirmwareVersionString);

		// Functions ported over - might not be needed.....
		void EnableRxInterrupt(void) { exar_port->IER_RG |= IER_RHR ; } ;
        void DisableRxInterrupt(void) { exar_port->IER_RG &= ~IER_RHR ;} ;
        void EnableTxInterrupt(void) { exar_port->IER_RG |= IER_THR ;} ;
        void DisableTxInterrupt(void) { exar_port->IER_RG &= ~IER_THR ; } ;
        void EnableBreakInterrupt(void) { ; } ;
        void DisableBreakInterrupt(void) { ; } ;
        void TxPolled(unsigned char Data)  {
            while ( !TxReady() )
                    ;
            WriteTxBuffer(Data);
        } ;
        void WriteBaudRate(unsigned long baud_rate) {  SetBaud((int)baud_rate) ; }  ;
        void WriteTxBuffer(unsigned char byt) { exar_port->THR_RG = byt; }    ;

        // from uart16c550
        void Dis485_Tx(void) 
		{
			exar_port->MCR_RG |= MCR_RTS;	// Active low....
		} ;
        void En485_Tx(void) 
		{
			exar_port->MCR_RG &= ~MCR_RTS;	// Active low....
		}  ;
        void Dis485_Rx(void)  
		{ 
			En485_Tx();
		} ;
        void En485_Rx(void) 
		{ 
			Dis485_Tx();
		} ;
        void ForceRTS(void)  
		{ 
			En485_Tx();
		}
        void ClearRTS(void)  
		{ 
			Dis485_Tx();
		}

        void EnHwHandShake(void)  { ;} ;
        void DisHwHandShake(void)  { ;} ;
        void EnSwHandShake(void)  { ;} ;
        void DisSwHandShake(void)  { ;} ;
		int CheckTxEmpty(void) {return (int)((exar_port->LSR_RG & LSR_TXE) ? 1 : 0) ; }

        void LoadXonXoff(void)  { exar_port->LCR_RG = lcr_copy;} ;
        void SetEFR(BYTE bits) { exar_port->LCR_RG = lcr_copy;} ;
        void ClrEFR(BYTE bits) { exar_port->LCR_RG = lcr_copy;} ;
        void WriteEFR(BYTE bits) { exar_port->LCR_RG = lcr_copy;} ;
        void WriteFCTR(BYTE bits) { exar_port->LCR_RG = lcr_copy;} ;
        void SetFCTR(BYTE bits) { exar_port->LCR_RG = lcr_copy;} ;
        void ClrFCTR(BYTE byt) { exar_port->LCR_RG = lcr_copy;} ;
        void WriteEMSR(BYTE bits){ exar_port->LCR_RG = lcr_copy;} ;
        void WriteFIFOctrl(BYTE byt = 0); // { exar_port->FCR_RG = FCR_ENABLE | ( byt & ( FCR_RST_RX | FCR_RST_TX )) ;} ;
        void ResetFIFO(BYTE byt = 0); // { exar_port->FCR_RG = FCR_ENABLE | ( byt & ( FCR_RST_RX | FCR_RST_TX )) ; } ;
        void SetRxIntLevel(int level) { exar_port->FCR_RG = FCR_ENABLE;} ;
        void SetTxIntLevel(int level) { exar_port->FCR_RG = FCR_ENABLE;};
		
		void StartRespondingToPolls()
		{
			m_bRespondToPoll = true;
		}
		void StopRespondingToPolls()
		{
			m_bRespondToPoll = false;
		}
		void DumpDMNetStats();
		void ResetDMNetStats();

        DmEncapChecksumHelper* m_pEncapChecksumHelper;
} ;

class CDMNetSlaveLISR : public LowLevelInterrupt
{
	private:
		CDMNetSlavePortLISRHandler *m_pPortSpecificLISRHandler[MAX_NUM_RECEIVERS];
	// Actual entry routine for derived LowLevelInterrupt objects.
	public:
		
		CDMNetSlaveLISR( int Vector ) ;
		void Entry();
		void SetPortLISRHandler(int,CDMNetSlavePortLISRHandler *);
};


class CDMNetSlavePortLISRHandler 
{
    friend class CDMUartCresnetSlave ;
    friend class CDMNetSlaveTimerLISR ;
    protected:
		CColdfire_BaseTimer          *   m_pTimer;   // network mode and timing parameters
		UART_REGS                *   m_pRegs;
       
        CDMUartCresnetSlave * m_pUartCF;
        CDMNetSlaveRxHISR * m_pRxHISR ;

        int m_iRxCount ;    // -1:
        int m_iRxExpected ; // -1:
        int m_iDoNotShipUp ;            // when TRUE, do not activate RxHISR

    public:
        int m_iWaitForResponseDelay ;
        CLinearBuffer *m_pTxBuf ;
        EXAR_16C684_PORT *exar_port ;

    public:
        CDMNetSlavePortLISRHandler(CDMUartCresnetSlave * pUartCF ) ;
		~CDMNetSlavePortLISRHandler();

        void  EntryRoutine();   // Actual LISR Handler Routine.
        void  SetStateRX(void) {
            m_pUartCF->DisableTxInt() ;
            m_pUartCF->m_RxIntBuf->Clear();
            m_iRxCount = 0 ;
            m_pUartCF->En485_Rx();
            m_iRxExpected = -1 ;
            m_pUartCF->m_iNetMode = STATE_WAIT_FOR_RX ;
            m_pTimer->ResetTimer() ;
            m_pUartCF->EnableRxInt() ;
        }  ;
        void  SetStateRXnoRXtoggle(void) {
            m_pUartCF->DisableTxInt() ;
            m_pUartCF->m_RxIntBuf->Clear();
            m_iRxCount = 0 ;
            m_iRxExpected = -1 ;
            m_pUartCF->m_iNetMode = STATE_WAIT_FOR_RX ;
            m_pTimer->ResetTimer() ;
            m_pUartCF->EnableRxInt() ;
        }  ;
        void ThrowPacketAway(void) ;
        void InvokeRxHISR(void) ;
} ;

class CDMNetSlaveTimerLISR  : public LowLevelInterrupt
{
    friend class CDMUartCresnetSlave ;
    friend class CDMNetSlavePortLISRHandler ;
    private:
        CColdfire_BaseTimer          *   m_pTimer;   // network mode and timing parameters
        UART_REGS                *   m_pRegs;
     
        CDMUartCresnetSlave * m_pUartCF;
        CDMNetSlaveRxHISR * m_pRxHISR ;
        CDMNetSlavePortLISRHandler    *   m_pSlaveLISRHandler;

    public:
        CDMNetSlaveTimerLISR(CDMUartCresnetSlave * pUartCF, int Vector ) ;

        void  Entry();   // Actual high level ISR.
};

class CDMNetSlaveRxHISR: public HighLevelInterrupt
{
    // friend class CDMUartCresnetSlave ;
    private:
        CColdfire_BaseTimer          *   m_pTimer;   // network mode and timing parameters
        UART_REGS                *   m_pRegs;
		
        CDMUartCresnetSlave           *m_pUartCF;
        
        CDMNetSlaveRxTask    *m_pRxTask ;
        CLinearBuffer *m_pTxBuf ;
        CREST_ALL_PACKETS    *m_pOut ;  // pointer to TxLISR ring buffer
        BYTE         m_iPrevTokenId ;    // for debug

    public:
        CREST_ALL_PACKETS m_pIn ;

        CDMNetSlaveRxHISR(CDMUartCresnetSlave * pOwnerUart);

        void  Entry();   // Actual high level ISR.

        int ProcessToken(void) ;
        int ProcessPnPidentifyRequest(void) ;
        int StartSlaveTxmit(int len = -1) ;
};

class CDMNetSlaveRxTask: public PipeTask
{
    public:
        CREST_ALL_PACKETS    m_pIn ;
        CREST_ALL_PACKETS    *m_pOut ;  // pointer to TxLISR ring buffer
        CDMUartCresnetSlave     *m_pUartCF;

        CDMNetSlaveRxTask(CDMUartCresnetSlave * pUartCF) ;
        ~CDMNetSlaveRxTask() ;

        void Entry() ;
} ;

// extern CDMUartCresnetSlave *pUartCresnetSlave[];
CDMUartCresnetSlave *GetCresnetSlaveUart(void);
CDMUartCresnetSlave *GetDMMidpointUart(UINT8 streamId);
// Are there any libraries specific header files
extern void HandleIncomingCresnetEncapsulatedPacket(CREST_ALL_PACKETS *);

// offsets into Midpoint buffer (CNET packt with encapsulation)
#define MIDPKT_START_ENCAP_LEN 1
#define MIDPKT_START_UNENCAP   3  // start of packet received



#endif // _DM_DMNET_SLAVE_H_


